(:
    // A sampling of date formats for RSS:
    //		Per item:
    //			none
    //			[Sep 9]
    //			<pubDate>Wed, 17 Sep 2003 13:30:00 PDT</pubDate>
    //			<dc:date>2003-08-26T07:15:15Z</dc:date>
    //			<dc:date>2003-08-15</dc:date>
    //		In <channel/>:
    //			<dc:date>2003-09-17T21:29:40+00:00</dc:date>
    //			<pubDate>Wed, 17 Sep 2003 02:35:33 PST</pubDate>
:)

declare namespace itms = "http://phobos.apple.com/rss/1.0/modules/itms/";
declare namespace rssLink = "http://purl.org/rss/1.0/modules/link/";

declare function local:atomContentString($node as element(), $strip) {
    let $mode := $node/@mode
    let $content := if ($mode = "base64") then
        afn:string-from-binary(xs:base64Binary($node/text()))
    else if ($mode = "escaped") then
        string($node) (: by parsing it becomes unescaped :)
    else (: default = xml :)
        if ($strip) then (: strip out tags and return text only :)
            string($node)
        else
            let $xml := for $child in $node/node()
                return afn:string-from-node($child)
            return string-join($xml, "")

    let $type := lower-case($node/@type)
    let $content := if ($type = "text/html" or $type = "html") then
        local:unescape($content)
    else $content
    return $content
};

declare function local:unescape($text as xs:string) {
    let $text := replace($text, "&amp;lt;", "&lt;")
    let $text := replace($text, "&amp;gt;", "&gt;")
    let $text := replace($text, "&amp;quot;", "&quot;")
    let $text := replace($text, "&amp;apos;", "&apos;")
    let $text := replace($text, "&amp;amp;", "&amp;")
    return $text
};

declare function local:escape($text as xs:string) {
    let $text := replace($text, "&amp;", "&amp;amp;")
    let $text := replace($text, "&lt;", "&amp;lt;")
    let $text := replace($text, "&gt;", "&amp;gt;")
    let $text := replace($text, "&quot;", "&amp;quot;")
    let $text := replace($text, "&apos;", "&amp;apos;")
    return $text
};

declare function local:stripMatchingTags($title as xs:string) {
    let $newTitle := replace($title, "&lt;([^> ]*?)(>|( [^>]*?>))(.*?)&lt;/\1>", "$4", "i")
    let $title := if ($title = $newTitle) then $title
        else local:stripMatchingTags($newTitle)
    return $title
};

declare function local:makeTitlePlainText($title as xs:string) {
    let $title := if (contains($title, "&lt;")) then
        let $strippedTitle := local:stripMatchingTags($title)
        (: strip out all <br>s and <hr>s :)
        let $strippedTitle := replace($strippedTitle, "&lt;(br|hr)\W*/?&gt;", " ", "i")
        let $strippedTitle := if (contains($strippedTitle, "&lt;img")) then        
            if (matches($strippedTitle, "&lt;img.*alt", "i")) then
                replace($strippedTitle, "&lt;img((.*alt=(&quot;|&apos;)?(.*?)\3?( .*?&gt;|&gt;))|(.*?&gt;))", "[$4]", "i")
                (: Only remove those img tags with attributes :)
            else replace($strippedTitle, "&lt;img .*?&gt;", "", "i")
        else $strippedTitle
        return $strippedTitle
    else $title

    (: Replace character references and entities; that "T" means "tidy" :)
    let $title := if (contains($title, "&amp;")) then    
        (: Don't let remaining tags be wiped out by tidy :)
        let $tidyTitle := replace($title, "&lt;(/?[^>]*?>)", "&amp;lt;$1")
        (: Wrap in HTML so that tidy uses UTF-8.  This is a temporary fix for 4086112 :)
        let $tidyTitle := concat("<html><head><meta http-equiv=""Content-Type"" content=""text/html; charset=UTF-8""><title>", $title , "</title></head></html>")
        return string(afn:node-from-string($tidyTitle, "T"))
    else $title
    return $title
};

(: Convert an RSS date format, such as Wed, 15 Dec 2004 08:14:10 -0800, to a W3 format, such as 2004-05-26T22:29:46-08:00 :)
declare function local:convertRSSDateToW3($date as xs:string) {
	let $month := replace($date, ".*((Jan)|(Feb)|(Mar)|(Apr)|(May)|(Jun)|(Jul)|(Aug)|(Sep)|(Oct)|(Nov)|(Dec)).*", "$1")
	let $month := if ($month = "Jan") then "01"
		else if ($month = "Feb") then "02"
		else if ($month = "Mar") then "03"
		else if ($month = "Apr") then "04"
		else if ($month = "May") then "05"
		else if ($month = "Jun") then "06"
		else if ($month = "Jul") then "07"
		else if ($month = "Aug") then "08"
		else if ($month = "Sep") then "09"
		else if ($month = "Oct") then "10"
		else if ($month = "Nov") then "11"
		else "12"
	let $date := replace($date, "[^,]+,\W*(\d+)\W*(\w+)\W*(\d+)\W*(\d{2}:\d{2}:\d{2})\W*(-|\+)(\d{1,2})(\d{2}$)", concat("$3-", $month, "-$1T$4$50$6:$7"))
	let $date := replace($date, "(-|\+)0(\d{2}):", "$1$2:")
	return $date
};

(: Resolves the char refs of the form &#123; to actual characters; does not handle hex char refs :)
declare function local:resolveCharacterReferences($string as xs:string) {
    let $prefix := substring-before($string, "&amp;#")
    let $result := if ($prefix = "") then $string else
        let $rest := substring($string, string-length($prefix) + 3)
        let $codepoint := substring-before($rest, ";")
        let $char := if (matches($codepoint, "^\d+$"))
            then codepoints-to-string((xs:integer($codepoint)))
            else ""
        let $converted := if ($char = "") then $string
            else concat($prefix, $char, local:resolveCharacterReferences(substring($rest, string-length($codepoint) + 2)))
        return $converted
    return $result
};

declare function local:enclosuresForElement($itemNode as element()) {
    let $atomNodes := $itemNode//link[@rel='enclosure']
    let $rssNodes  := $itemNode//enclosure

    for $node in $atomNodes|$rssNodes
        (: Atom 1.0 enclosure format <enclosure href="" type="" length=""> :)
        let $url    := string($node/@href)
        let $type   := string($node/@type)
        let $length := string($node/@length)

        (: RSS 2.0 enclosure format <enclosure url="" type="" length=""> :)
        let $url  := if ($url)  then $url  else string($node/@url)

        return element enclosure {
        element url    {$url},
        element type   {$type},
        element length {$length}
        }
};
	
let $feed := .
let $channel := $feed/*[1]

(: A feed element denotes an Atom feed, versus a channel element for RSS :)
let $isAtom := if ($feed[name() = "feed"]) then 1 else 0

let $version := number($feed/@version)
let $version := if ($version) then $version else
    if (namespace-uri-for-prefix("", $feed) = xs:anyURI("http://purl.org/rss/1.0/")) then 1.0 else ()
let $version := if ($version) then $version else 0
let $needsEscape := if ($isAtom) then 0
    else 
        let $escape := if ($version < 0.92) then
            (: Check to see if it looks like html :)
            let $descriptions := string-join($feed//*:description, "")    
            let $matches := matches($descriptions, "(&lt;\w+(\s+\w+(\s*=\s*('|"")?[^\4]+\4)?)*\s*/?&gt;)|(&amp;\w+;)|(&amp;\w+;)")
            (: default for 0.9, 0.91 is plain text, so if it looks like html don't escape it :)
            return if ($matches) then 0 else 1
        else 0
    return $escape

let $feedElements := if ($isAtom) then
    let $title := local:makeTitlePlainText(local:atomContentString($feed/title, 1))
    let $description := local:atomContentString($feed/tagline, 0)
    let $feedLink := string($feed/link[@rel="alternate"]/@href)
    let $scheme := lower-case(substring-before($feedLink, ":"))
    let $feedLink := if ($scheme = "javascript" or $scheme = "file") then () else $feedLink
    let $date := string($feed/modified/text())
    return (
        element title {normalize-space($title)},
	element description {$description},
	element link {escape-uri(normalize-space($feedLink), false())},
	element date {normalize-space($date)},
        element imageTitle {()},
	element imageURL {()},
	element imageLink {()},
	element imageWidth {()},
	element imageHeight {()}
    )
else 
    let $title := string-join($channel/*:title[1]/text(), "")
    let $description := string-join($channel/*:description[1]/text(), "")
    let $feedLink := string($channel/*:link[1]/text())
    let $scheme := lower-case(substring-before($feedLink, ":"))
    let $feedLink := if ($scheme = "javascript" or $scheme = "file") then () else $feedLink
    let $date := string($channel/*:date[1]/text())
    let $date := if ($date) then $date else string($channel/lastBuildDate/text())
    let $date := if ($date) then $date else string($channel/pubDate/text())
    let $image := $channel/image
    let $imageTitle := string-join($image/title/text(), "")
    let $imageURL := string($image/url/text())
    let $imageLink := string($image/link/text())
    let $imageWidth := string($image/width/text())
    let $imageHeight := string($image/height/text())
    return (
        element title {normalize-space($title)},
	element description {$description},
	element link {escape-uri(normalize-space($feedLink), false())},
	element date {normalize-space($date)},
	element imageTitle {normalize-space($imageTitle)},
	element imageURL {normalize-space($imageURL)},
	element imageLink {escape-uri(normalize-space($imageLink), false())},
	element imageWidth {normalize-space($imageWidth)},
	element imageHeight {normalize-space($imageHeight)}
    )

let $allItems := if ($isAtom) then $feed//entry else .//item
let $feedItems := if ($isAtom) then for $rawItem at $i in $allItems
    	let $title := local:makeTitlePlainText(local:atomContentString($rawItem/title, 1))
	let $author := string($rawItem/author/name/text())
        let $author := if ($author) then $author else string($feed/author/name/text())
	let $link := string($rawItem/link[@rel="alternate"]/@href)
        let $scheme := lower-case(substring-before($link, ":"))
        let $link := if ($scheme = "javascript" or $scheme = "file") then () else $link
        let $guid := string($rawItem/id/text())
        let $date := if( $rawItem/published) then string($rawItem/published/text())
                else if( $rawItem/updated) then string($rawItem/updated/text())
                else if( $rawItem/created ) then string($rawItem/created/text())
                else string($rawItem/modified/text())
	let $summary := local:atomContentString($rawItem/summary, 0)
        let $content := for $contentNode in $rawItem/content
            let $contentType := string($contentNode/@type)
            let $content := if ($contentType = "multipart/alternate") then
                (: prefer marked up version :)
                let $content := $contentNode/content[@type = "application/xhtml+xml"
                    or @type = "text/html" or @mode = "xml" or @mode = "escaped"][1]
                (: otherwise look for text :)
                let $content := if ($content) then $content else
                    $contentNode/content[@type = "text/plain" or not(@type)][1]
                return $content
            else $contentNode
            return local:atomContentString($content, 0)
        let $content := string-join($content, "")
	let $description := $summary
	let $enclosures := 	local:enclosuresForElement($rawItem)
	let $xmlSource := afn:string-from-node($rawItem)

	(: order is important -- check enum definitions :)
        let $item := element item {
            element title {normalize-space($title)},
            element creator {normalize-space($author)},
            element link {escape-uri(normalize-space($link), false())},
            element content {$content},
            element date {normalize-space($date)},
            element guid {normalize-space($guid)},
            element description {$description},
			element enclosures {$enclosures},
            element xml {$rawItem},
            element xmlSource {$xmlSource},
            $i (: to preserve source order :)
	}
        order by $item/guid
	return $item
    else for $rawItem at $i in $allItems
	let $itemTitle := string-join($rawItem/*:title[1]/text(), "")
        (: Strip out matching tags and replace img tags with their alt text. :)
        let $itemTitle := local:makeTitlePlainText($itemTitle)
	let $creator := string($rawItem/author/text())
        let $creator := if ($creator) then $creator else string($rawItem/*:creator[1]/text())
        let $creator := if (contains($creator, "&amp;#")) then local:resolveCharacterReferences($creator) else $creator
        let $guid := string($rawItem/guid/text())
        let $guid := if ($guid) then $guid else string($rawItem/rssLink:link[@rssLink:rel="http://purl.org/rss/1.0/modules/link/#permalink"]/@*:resource)
        let $guid := if ($guid) then $guid else string($rawItem/rssLink:permalink/@*:resource)        
	let $link := string-join($rawItem/*:link[1]/text(), "")
	(: To fix Ken Bereskin's Radio blog :)
	let $link := if ($link) then $link else $guid
        let $scheme := lower-case(substring-before($link, ":"))
	let $link := if ($scheme = "javascript" or $scheme = "file") then () else $link
	(: Handle iTMS items differently, to reformat them :)
	let $iTMS := contains($feedLink, "phobos.apple.com")
	let $content := $rawItem/*:encoded[1]/text()
    let $date := string($rawItem/*:date[1]/text())
	let $date := if ($date) then $date else string($rawItem/pubDate/text())
        let $date := if (not($iTMS)) then $date else
	    let $rank := $rawItem/itms:rank/text()
	    let $date := if (not($rank)) then $date else
                let $rank := normalize-space($rank)
                let $date := xs:dateTime(local:convertRSSDateToW3($date))
                let $date := $date + xdt:dayTimeDuration(concat("-PT", $rank, "M"))
                return xs:string($date)
            return $date
	let $descNode := $rawItem/*:description[1]
	let $desc := string-join($descNode/text(), "")
	let $desc := if ($needsEscape or $descNode/@type = "text/plain") then 
		local:escape($desc)
            else $desc
	let $enclosures := local:enclosuresForElement($rawItem)
	let $xmlSource := afn:string-from-node($rawItem)

	(: order is important -- check enum definitions :)
        let $item := element item {
            element title {normalize-space($itemTitle)},
            element creator {normalize-space($creator)},
            element link {escape-uri(normalize-space($link), false())},
            element content {$content},
            element date {normalize-space($date)},
            element guid {normalize-space($guid)},
            element description {$desc},
            element enclosures {$enclosures},
            element xml {$rawItem},			
            element xmlSource {$xmlSource},
            $i (: to preserve source order :)
        }
        order by $item/guid
	return $item

(: Weed out duplicates based on guid. At this point the elements are sorted by guid, so we keep every one that has an empty guid or who's guid doesn't match the previous one :)
let $feedItems := for $feedItem at $i in $feedItems
    let $guid := $feedItem/guid
    where $i = 1 or ($guid/text() = "" or $guid != $feedItems[$i - 1]/guid)
    order by number($feedItem/text()) (: sort back to original xml order :)
    return $feedItem
    
(: order is important -- check enum definitions :)		
return element feed {
    $feedElements,
    element items {$feedItems},
    element xml {$channel}	
}
